<?php
// Iniciar sessão para evitar duplicação ao recarregar
session_start();

require_once 'auth.php'; // Proteção de login
require_once 'config.php';
require_once 'phpqrcode/qrlib.php';

$pdo = getDBConnection();

// Processar geração com PRG Pattern (Post-Redirect-Get)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['gerar'])) {
    $quantidade = (int)$_POST['quantidade'];
    
    if ($quantidade < 1 || $quantidade > 1000) {
        $_SESSION['msg_erro'] = "⚠️ Quantidade deve ser entre 1 e 1000!";
        header('Location: ' . $_SERVER['PHP_SELF']);
        exit;
    } else {
        try {
            // Criar diretório se não existir
            $dir_qr = 'qrcodes/';
            if (!file_exists($dir_qr)) {
                mkdir($dir_qr, 0777, true);
            }

            $pdo->beginTransaction();
            
            // Pegar último número
            $stmt = $pdo->query("SELECT COALESCE(MAX(id), 0) as ultimo FROM convidados");
            $ultimo = $stmt->fetch()['ultimo'];
            
            $gerados = 0;
            
            for ($i = 1; $i <= $quantidade; $i++) {
                $numero = $ultimo + $i;
                
                // Gerar hash único
                $hash = gerarHash($numero);
                
                // Gerar identificação completa
                $identificacao = gerarIdentificacao($numero);
                $qr_file_name = 'qr_' . $identificacao . '.png';
                $qr_file_path = $dir_qr . $qr_file_name;
                
                // URL de confirmação
                $url_confirmacao = URL_BASE . '/confirmar.php?id=' . $numero;
                
                // Gerar QR code com logo e identificação
                gerarQRCodeComLogo($url_confirmacao, $qr_file_path, 'logoo.png', $identificacao);
                
                // Inserir no banco
                $stmt = $pdo->prepare("
                    INSERT INTO convidados (nome, qrcode_hash, qrcode_image, confirmado, presente, compor_mesa) 
                    VALUES (NULL, :hash, :image, 0, 0, 0)
                ");
                
                $stmt->execute([
                    ':hash' => $hash,
                    ':image' => $qr_file_name
                ]);
                
                $gerados++;
            }
            
            $pdo->commit();
            $primeiro = formatarNumero($ultimo + 1);
            $ultimo_num = formatarNumero($ultimo + $quantidade);
            
            // Salvar mensagem na sessão e redirecionar (evita resubmit)
            $_SESSION['msg_sucesso'] = "✅ {$gerados} QR codes gerados! (#{$primeiro} até #{$ultimo_num})<br>Agora vá para o painel administrativo para atribuir nomes.";
            header('Location: ' . $_SERVER['PHP_SELF']);
            exit;
            
        } catch (Exception $e) {
            $pdo->rollBack();
            $_SESSION['msg_erro'] = "❌ Erro ao gerar QR codes: " . $e->getMessage();
            header('Location: ' . $_SERVER['PHP_SELF']);
            exit;
        }
    }
}

// Recuperar mensagens da sessão
$msg_sucesso = $_SESSION['msg_sucesso'] ?? null;
$msg_erro = $_SESSION['msg_erro'] ?? null;

// Limpar mensagens após exibir
unset($_SESSION['msg_sucesso'], $_SESSION['msg_erro']);

// Processar download de QR codes
if (isset($_GET['download']) && isset($_GET['id'])) {
    try {
        $id = (int)$_GET['id'];
        
        $stmt = $pdo->prepare("SELECT qrcode_image FROM convidados WHERE id = :id");
        $stmt->execute([':id' => $id]);
        $convidado = $stmt->fetch();
        
        if ($convidado && $convidado['qrcode_image']) {
            $qr_file_path = 'qrcodes/' . $convidado['qrcode_image'];
            
            if (file_exists($qr_file_path)) {
                header('Content-Type: image/png');
                header('Content-Disposition: attachment; filename="' . $convidado['qrcode_image'] . '"');
                header('Content-Length: ' . filesize($qr_file_path));
                readfile($qr_file_path);
                exit;
            }
        }
        
        die("QR Code não encontrado!");
        
    } catch (Exception $e) {
        die("Erro ao baixar QR Code: " . $e->getMessage());
    }
}

// Processar download em massa
if (isset($_POST['download_massa']) && isset($_POST['qrcodes_selecionados'])) {
    try {
        $ids = $_POST['qrcodes_selecionados'];
        
        $zip = new ZipArchive();
        $zip_name = 'qrcodes_' . date('Y-m-d_H-i-s') . '.zip';
        $zip_path = 'qrcodes/' . $zip_name;
        
        if ($zip->open($zip_path, ZipArchive::CREATE) === TRUE) {
            foreach ($ids as $id) {
                $stmt = $pdo->prepare("SELECT qrcode_image FROM convidados WHERE id = :id");
                $stmt->execute([':id' => $id]);
                $convidado = $stmt->fetch();
                
                if ($convidado && $convidado['qrcode_image']) {
                    $qr_file_path = 'qrcodes/' . $convidado['qrcode_image'];
                    if (file_exists($qr_file_path)) {
                        $zip->addFile($qr_file_path, $convidado['qrcode_image']);
                    }
                }
            }
            
            $zip->close();
            
            header('Content-Type: application/zip');
            header('Content-Disposition: attachment; filename="' . $zip_name . '"');
            header('Content-Length: ' . filesize($zip_path));
            readfile($zip_path);
            unlink($zip_path);
            exit;
        }
        
    } catch (Exception $e) {
        $_SESSION['msg_erro'] = "Erro ao gerar arquivo ZIP: " . $e->getMessage();
        header('Location: ' . $_SERVER['PHP_SELF']);
        exit;
    }
}

// Função para gerar QR Code com logo, identificação e cabeçalho "CONFIRME" GRANDE
function gerarQRCodeComLogo($text, $outfile, $logo_path, $identificacao) {
    // Gerar QR code temporário
    QRcode::png($text, $outfile, QR_ECLEVEL_H, 10, 2);

    $qrcode = imagecreatefrompng($outfile);
    $qr_width = imagesx($qrcode);
    $qr_height = imagesy($qrcode);

    // Aplicar fundo branco no QR code
    $white_bg = imagecreatetruecolor($qr_width, $qr_height);
    $white = imagecolorallocate($white_bg, 255, 255, 255);
    imagefill($white_bg, 0, 0, $white);
    imagecopy($white_bg, $qrcode, 0, 0, 0, 0, $qr_width, $qr_height);
    $qrcode = $white_bg;

    // Adicionar logo no centro do QR code
    $logo = imagecreatefrompng($logo_path);
    $logo_width = imagesx($logo);
    $logo_height = imagesy($logo);

    $new_logo_width = $qr_width * 0.20;
    $new_logo_height = ($logo_height / $logo_width) * $new_logo_width;

    $logo_resized = imagecreatetruecolor($new_logo_width, $new_logo_height);
    imagealphablending($logo_resized, false);
    imagesavealpha($logo_resized, true);
    imagecopyresampled($logo_resized, $logo, 0, 0, 0, 0, $new_logo_width, $new_logo_height, $logo_width, $logo_height);

    $logo_bg_size = $new_logo_width + 20;
    $logo_bg = imagecreatetruecolor($logo_bg_size, $logo_bg_size);
    $white = imagecolorallocate($logo_bg, 255, 255, 255);
    imagefill($logo_bg, 0, 0, $white);

    $x_logo = ($qr_width - $logo_bg_size) / 2;
    $y_logo = ($qr_height - $logo_bg_size) / 2;

    imagecopy($qrcode, $logo_bg, $x_logo, $y_logo, 0, 0, $logo_bg_size, $logo_bg_size);

    $x_logo_final = ($qr_width - $new_logo_width) / 2;
    $y_logo_final = ($qr_height - $new_logo_height) / 2;
    imagecopy($qrcode, $logo_resized, $x_logo_final, $y_logo_final, 0, 0, $new_logo_width, $new_logo_height);

    // Adicionar identificação no canto inferior direito
    $font_size = 3;
    $font_color = imagecolorallocate($qrcode, 80, 80, 80);
    
    $text_width = imagefontwidth($font_size) * strlen($identificacao);
    $text_height = imagefontheight($font_size);
    
    $x_text = $qr_width - $text_width - 10;
    $y_text = $qr_height - $text_height - 10;
    
    $bg_padding = 5;
    $bg_color = imagecolorallocatealpha($qrcode, 255, 255, 255, 30);
    imagefilledrectangle($qrcode, 
        $x_text - $bg_padding, 
        $y_text - $bg_padding, 
        $x_text + $text_width + $bg_padding, 
        $y_text + $text_height + $bg_padding, 
        $bg_color
    );
    
    imagestring($qrcode, $font_size, $x_text, $y_text, $identificacao, $font_color);

    // ============================================================
    // ADICIONAR CABEÇALHO "CONFIRME" NO TOPO - VERSÃO AUMENTADA
    // ============================================================
    
    // AUMENTADO: Altura do cabeçalho agora é 30% (era 20%)
    $header_height = intval($qr_width * 0.12); // 35% da largura do QR code (AUMENTADO!)
    $padding = 15; // Espaçamento
    
    // Criar nova imagem com espaço para cabeçalho
    $final_width = $qr_width + ($padding * 2);
    $final_height = $qr_height + $header_height + ($padding * 2) + 10;
    
    $final_image = imagecreatetruecolor($final_width, $final_height);
    
    // Cores
    $white = imagecolorallocate($final_image, 255, 255, 255);
    $black = imagecolorallocate($final_image, 0, 0, 0);
    
    // Preencher fundo branco
    imagefill($final_image, 0, 0, $white);
    
    // Desenhar borda preta
    
    // Adicionar texto "CONFIRME" no topo
    $text = "CONFIRME";
    
    // Usar fonte TrueType se disponível, senão usar fonte GD padrão
    $font_path = '/usr/share/fonts/truetype/dejavu/DejaVuSans-Bold.ttf';
    
    if (file_exists($font_path)) {
        // AUMENTADO: Tamanho da fonte agora é 14% (era 8%)
        $font_size_tt = intval($qr_width * 0.12);
        
        // Obter dimensões do texto
        $bbox = imagettfbbox($font_size_tt, 0, $font_path, $text);
        $text_width = abs($bbox[4] - $bbox[0]);
        $text_height = abs($bbox[5] - $bbox[1]);
        
        // Centralizar texto
        $x = ($final_width - $text_width) / 2;
        $y = $padding + $text_height;
        
        // Desenhar texto
        imagettftext($final_image, $font_size_tt, 0, $x, $y, $black, $font_path, $text);
    } else {
        // Fallback: usar fonte GD com tamanho 5 (maior disponível)
        // Desenhar MÚLTIPLAS VEZES para simular um texto maior e mais grosso
        $gd_font_size = 5;
        $text_width = imagefontwidth($gd_font_size) * strlen($text);
        $text_height = imagefontheight($gd_font_size);
        
        // Escala para simular texto maior (desenhar 3x maior)
        $scale = 3;
        
        $x_base = ($final_width - ($text_width * $scale)) / 2;
        $y_base = $border_width + $padding + ($header_height - ($text_height * $scale)) / 2;
        
        // Desenhar cada letra aumentada
        for ($char_idx = 0; $char_idx < strlen($text); $char_idx++) {
            $char = $text[$char_idx];
            $char_width = imagefontwidth($gd_font_size);
            
            // Posição base desta letra
            $x_char_base = $x_base + ($char_idx * $char_width * $scale);
            
            // Desenhar a letra várias vezes para criar efeito de escala e negrito
            for ($sy = 0; $sy < $scale; $sy++) {
                for ($sx = 0; $sx < $scale; $sx++) {
                    $x_char = $x_char_base + ($sx * $char_width);
                    $y_char = $y_base + ($sy * $text_height);
                    imagestring($final_image, $gd_font_size, $x_char, $y_char, $char, $black);
                }
            }
        }
    }
    
    // Copiar QR code para a imagem final (abaixo do cabeçalho)
    $qr_x = $padding;
    $qr_y = $header_height + $padding;
    
    imagecopy($final_image, $qrcode, $qr_x, $qr_y, 0, 0, $qr_width, $qr_height);

    // Salvar imagem final
    imagepng($final_image, $outfile);
    
    // Limpar memória
    imagedestroy($qrcode);
    imagedestroy($logo);
    imagedestroy($logo_resized);
    imagedestroy($logo_bg);
    imagedestroy($final_image);
}

// Buscar todos os QR codes
$todos_qrcodes = $pdo->query("
    SELECT id, nome, qrcode_hash, qrcode_image, confirmado, presente, compor_mesa 
    FROM convidados 
    ORDER BY id ASC
")->fetchAll();

// Estatísticas
$stats = $pdo->query("
    SELECT 
        COUNT(*) as total,
        SUM(CASE WHEN nome IS NULL THEN 1 ELSE 0 END) as disponiveis,
        SUM(CASE WHEN nome IS NOT NULL AND confirmado = 0 THEN 1 ELSE 0 END) as atribuidos,
        SUM(CASE WHEN confirmado = 1 THEN 1 ELSE 0 END) as confirmados,
        SUM(CASE WHEN presente = 1 THEN 1 ELSE 0 END) as presentes
    FROM convidados
")->fetch();
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gerar QR Codes - Evento Posse</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; background: linear-gradient(135deg, #0a1465 0%, #081050 100%); min-height: 100vh; padding: 20px; }
        .container { max-width: 1200px; margin: 0 auto; }
        .header { text-align: center; color: white; margin-bottom: 30px; }
        .header h1 { font-size: 32px; margin-bottom: 10px; }
        .header p { font-size: 16px; opacity: 0.9; }
        .card { background: white; border-radius: 20px; padding: 40px; box-shadow: 0 20px 60px rgba(0,0,0,0.3); margin-bottom: 20px; }
        .stats { display: grid; grid-template-columns: repeat(auto-fit, minmax(140px, 1fr)); gap: 15px; margin-bottom: 30px; }
        .stat-box { text-align: center; padding: 15px; background: #f8f9fa; border-radius: 12px; border-left: 4px solid #0a1465; }
        .stat-box h3 { font-size: 13px; color: #666; margin-bottom: 8px; }
        .stat-box .number { font-size: 28px; font-weight: bold; color: #0a1465; }
        .alert { padding: 15px 20px; border-radius: 10px; margin-bottom: 20px; font-weight: 500; }
        .alert-success { background: #d4edda; color: #155724; border: 1px solid #c3e6cb; }
        .alert-error { background: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; }
        .form-group { margin-bottom: 25px; }
        .form-group label { display: block; margin-bottom: 8px; color: #333; font-weight: 600; font-size: 14px; }
        .form-group input { width: 100%; padding: 15px; border: 2px solid #ddd; border-radius: 10px; font-size: 16px; transition: all 0.3s; }
        .form-group input:focus { outline: none; border-color: #0a1465; }
        .info-box { background: #e7f3ff; border-left: 4px solid #2196F3; padding: 20px; margin-bottom: 25px; border-radius: 8px; }
        .info-box h3 { color: #1976D2; margin-bottom: 10px; font-size: 16px; }
        .info-box ul { list-style: none; padding: 0; }
        .info-box li { padding: 6px 0; color: #555; font-size: 14px; }
        .info-box li:before { content: "✓ "; color: #2196F3; font-weight: bold; margin-right: 8px; }
        .btn { padding: 15px 30px; border: none; border-radius: 12px; font-size: 16px; font-weight: 600; cursor: pointer; transition: all 0.3s; text-decoration: none; display: inline-block; text-align: center; }
        .btn-primary { background: linear-gradient(135deg, #0a1465 0%, #081050 100%); color: white; width: 100%; }
        .btn-primary:hover { transform: translateY(-2px); box-shadow: 0 10px 25px rgba(10, 20, 101, 0.4); }
        .btn-secondary { background: #6c757d; color: white; width: 100%; margin-top: 10px; }
        .btn-secondary:hover { background: #5a6268; }
        .btn-success { background: #28a745; color: white; font-size: 14px; padding: 8px 16px; }
        .btn-success:hover { background: #218838; }
        .btn-download { background: #17a2b8; color: white; font-size: 13px; padding: 8px 16px; margin-right: 5px; }
        .btn-download:hover { background: #138496; }
        .warning-box { background: #fff3cd; border: 2px solid #ffc107; border-radius: 10px; padding: 20px; margin-bottom: 20px; }
        .warning-box h3 { color: #856404; margin-bottom: 10px; }
        .warning-box p { color: #856404; line-height: 1.6; margin-bottom: 10px; }
        .qrcode-list { margin-top: 30px; }
        .qrcode-list h2 { margin-bottom: 20px; color: #333; }
        .qrcode-grid { display: grid; grid-template-columns: repeat(auto-fill, minmax(250px, 1fr)); gap: 20px; margin-top: 20px; }
        .qrcode-item { background: #f8f9fa; border-radius: 12px; padding: 15px; text-align: center; border: 2px solid #e0e0e0; transition: all 0.3s; }
        .qrcode-item:hover { border-color: #0a1465; box-shadow: 0 5px 15px rgba(10, 20, 101, 0.2); }
        .qrcode-item img { width: 100%; max-width: 200px; height: auto; border-radius: 8px; margin-bottom: 10px; }
        .qrcode-item h4 { font-size: 14px; color: #333; margin-bottom: 8px; }
        .qrcode-item .status { font-size: 12px; padding: 4px 8px; border-radius: 6px; display: inline-block; margin-bottom: 10px; }
        .status.disponivel { background: #d4edda; color: #155724; }
        .status.atribuido { background: #fff3cd; color: #856404; }
        .status.confirmado { background: #d1ecf1; color: #0c5460; }
        .checkbox-container { margin: 10px 0; }
        .checkbox-container input { width: auto; margin-right: 5px; }
        .action-buttons { display: flex; gap: 10px; justify-content: center; flex-wrap: wrap; }
        .select-all-container { margin-bottom: 20px; padding: 15px; background: #e7f3ff; border-radius: 10px; display: flex; justify-content: space-between; align-items: center; }
        .select-all-container label { font-weight: 600; color: #333; }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>Gerador de QR Codes</h1>
            <p>Ingressos Numerados - Evento Posse</p>
        </div>
        
        <div class="card">
            <?php if ($msg_sucesso): ?>
                <div class="alert alert-success"><?= $msg_sucesso ?></div>
            <?php endif; ?>
            
            <?php if ($msg_erro): ?>
                <div class="alert alert-error"><?= $msg_erro ?></div>
            <?php endif; ?>
            
            <div class="stats">
                <div class="stat-box">
                    <h3>Total</h3>
                    <div class="number"><?= $stats['total'] ?></div>
                </div>
                <div class="stat-box">
                    <h3>Disponíveis</h3>
                    <div class="number" style="color: #28a745;"><?= $stats['disponiveis'] ?></div>
                </div>
                <div class="stat-box">
                    <h3>Atribuídos</h3>
                    <div class="number" style="color: #ffc107;"><?= $stats['atribuidos'] ?></div>
                </div>
                <div class="stat-box">
                    <h3>Confirmados</h3>
                    <div class="number" style="color: #17a2b8;"><?= $stats['confirmados'] ?></div>
                </div>
                <div class="stat-box">
                    <h3>Presentes</h3>
                    <div class="number" style="color: #0a1465;"><?= $stats['presentes'] ?></div>
                </div>
            </div>

            <div class="warning-box">
                <h3>📌 Como Funciona:</h3>
                <p><strong>1.</strong> Aqui você gera os QR codes com números únicos</p>
                <p><strong>2.</strong> Cada QR code terá uma identificação (data-número)</p>
                <p><strong>3.</strong> Você pode baixar individualmente ou em massa</p>
                <p><strong>4.</strong> Depois vá para o Painel Administrativo para atribuir nomes</p>
            </div>
          
            
            <form method="POST">
                <div class="form-group">
                    <label>📊 Quantos QR codes deseja gerar?</label>
                    <input type="number" name="quantidade" min="1" max="1000" value="50" required>
                </div>
                
                <button type="submit" name="gerar" class="btn btn-primary">
                     Gerar QR Codes
                </button>
            </form>
            
            <button onclick="window.open('admin.php', '_blank')" class="btn btn-secondary">
                 Ir para Painel Administrativo
            </button>
        </div>

        <?php if (!empty($todos_qrcodes)): ?>
        <div class="card">
            <div class="qrcode-list">
                <h2>📋 QR Codes Gerados (<?= count($todos_qrcodes) ?>)</h2>
                
                <form method="POST" id="formDownloadMassa">
                    <div class="select-all-container">
                        <label>
                            <input type="checkbox" id="selectAll" onclick="toggleSelectAll(this)">
                            Selecionar Todos
                        </label>
                        <button type="submit" name="download_massa" class="btn btn-success" id="btnDownloadMassa" disabled>
                            📥 Download em Massa (ZIP)
                        </button>
                    </div>

                    <div class="qrcode-grid">
                        <?php foreach ($todos_qrcodes as $qr): ?>
                            <?php
                                $identificacao = pathinfo($qr['qrcode_image'], PATHINFO_FILENAME);
                                $identificacao = str_replace('qr_', '', $identificacao);
                                
                                if ($qr['nome']) {
                                    $status = $qr['confirmado'] ? 'confirmado' : 'atribuido';
                                    $status_text = $qr['confirmado'] ? 'Confirmado' : 'Atribuído';
                                } else {
                                    $status = 'disponivel';
                                    $status_text = 'Disponível';
                                }
                            ?>
                            <div class="qrcode-item">
                                <img src="qrcodes/<?= htmlspecialchars($qr['qrcode_image']) ?>" alt="QR Code">
                                <h4>ID: <?= str_pad($qr['id'], 4, '0', STR_PAD_LEFT) ?></h4>
                                <p style="font-size: 12px; color: #666; margin-bottom: 5px;">
                                    <?= htmlspecialchars($identificacao) ?>
                                </p>
                                <?php if ($qr['nome']): ?>
                                    <p style="font-size: 13px; font-weight: 600; color: #333; margin-bottom: 5px;">
                                        <?= htmlspecialchars($qr['nome']) ?>
                                    </p>
                                <?php endif; ?>
                                <span class="status <?= $status ?>"><?= $status_text ?></span>
                                
                                <div class="checkbox-container">
                                    <label>
                                        <input type="checkbox" name="qrcodes_selecionados[]" 
                                               value="<?= $qr['id'] ?>" 
                                               class="qr-checkbox"
                                               onchange="updateDownloadButton()">
                                        Selecionar
                                    </label>
                                </div>
                                
                                <div class="action-buttons">
                                    <a href="?download=1&id=<?= $qr['id'] ?>" 
                                       class="btn btn-download">
                                        💾 Download
                                    </a>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </form>
            </div>
        </div>
        <?php endif; ?>
    </div>

    <script>
        function toggleSelectAll(checkbox) {
            const checkboxes = document.querySelectorAll('.qr-checkbox');
            checkboxes.forEach(cb => cb.checked = checkbox.checked);
            updateDownloadButton();
        }

        function updateDownloadButton() {
            const checkboxes = document.querySelectorAll('.qr-checkbox:checked');
            const btnDownload = document.getElementById('btnDownloadMassa');
            btnDownload.disabled = checkboxes.length === 0;
            
            if (checkboxes.length > 0) {
                btnDownload.textContent = `📥 Download em Massa (${checkboxes.length} selecionados)`;
            } else {
                btnDownload.textContent = '📥 Download em Massa (ZIP)';
            }
        }

        document.getElementById('formDownloadMassa').addEventListener('submit', function(e) {
            const checkboxes = document.querySelectorAll('.qr-checkbox:checked');
            if (checkboxes.length === 0) {
                e.preventDefault();
                alert('Selecione pelo menos um QR code para download!');
                return false;
            }
            
            if (!confirm(`Deseja baixar ${checkboxes.length} QR codes em um arquivo ZIP?`)) {
                e.preventDefault();
                return false;
            }
        });
    </script>
</body>
</html>
