<?php
require_once 'auth.php';
require_once 'config.php';

$pdo = getDBConnection();

// Processamento de ações
if (isset($_POST['acao']) && $_POST['acao'] === 'download_massa' && isset($_POST['ids'])) {
    $ids = json_decode($_POST['ids'], true);
    if (!empty($ids)) {
        try {
            $zip = new ZipArchive();
            $zip_name = 'qrcodes_' . date('Y-m-d_H-i-s') . '.zip';
            $zip_path = 'qrcodes/' . $zip_name;
            if ($zip->open($zip_path, ZipArchive::CREATE) === TRUE) {
                $placeholders = str_repeat('?,', count($ids) - 1) . '?';
                $stmt = $pdo->prepare("SELECT id, qrcode_image FROM convidados WHERE id IN ($placeholders)");
                $stmt->execute($ids);
                while ($row = $stmt->fetch()) {
                    if ($row['qrcode_image']) {
                        $qr_file_path = 'qrcodes/' . $row['qrcode_image'];
                        if (file_exists($qr_file_path)) {
                            $new_name = 'QR_' . str_pad($row['id'], 4, '0', STR_PAD_LEFT) . '.png';
                            $zip->addFile($qr_file_path, $new_name);
                        }
                    }
                }
                $zip->close();
                header('Content-Type: application/zip');
                header('Content-Disposition: attachment; filename="' . $zip_name . '"');
                header('Content-Length: ' . filesize($zip_path));
                readfile($zip_path);
                unlink($zip_path);
                exit;
            }
        } catch (Exception $e) {
            $msg_erro = "Erro ao gerar ZIP: " . $e->getMessage();
        }
    }
}

if (isset($_POST['acao']) && $_POST['acao'] === 'apagar_massa' && isset($_POST['ids'])) {
    $ids = json_decode($_POST['ids'], true);
    if (!empty($ids)) {
        try {
            $pdo->beginTransaction();
            $placeholders = str_repeat('?,', count($ids) - 1) . '?';
            $stmt = $pdo->prepare("SELECT qrcode_image FROM convidados WHERE id IN ($placeholders)");
            $stmt->execute($ids);
            $deleted_files = 0;
            while ($row = $stmt->fetch()) {
                if ($row['qrcode_image']) {
                    $qr_file_path = 'qrcodes/' . $row['qrcode_image'];
                    if (file_exists($qr_file_path)) {
                        unlink($qr_file_path);
                        $deleted_files++;
                    }
                }
            }
            $stmt = $pdo->prepare("DELETE FROM convidados WHERE id IN ($placeholders)");
            $stmt->execute($ids);
            $pdo->commit();
            $msg_sucesso = "✅ " . count($ids) . " QR codes deletados! ($deleted_files arquivos removidos)";
        } catch (Exception $e) {
            $pdo->rollBack();
            $msg_erro = "❌ Erro ao deletar: " . $e->getMessage();
        }
    }
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['acao']) && $_POST['acao'] === 'atribuir') {
        $id = $_POST['id'];
        $nome = trim($_POST['nome']);
        $compor_mesa = isset($_POST['compor_mesa']) ? 1 : 0;
        $stmt = $pdo->prepare("UPDATE convidados SET nome = :nome, compor_mesa = :compor_mesa, data_cadastro = CASE WHEN nome IS NULL THEN NOW() ELSE data_cadastro END WHERE id = :id");
        $stmt->execute([':nome' => $nome, ':compor_mesa' => $compor_mesa, ':id' => $id]);
        $msg_sucesso = "✅ Ingresso #" . formatarNumero($id) . " atualizado!";
    }
    
    if (isset($_POST['acao']) && $_POST['acao'] === 'limpar') {
        $id = $_POST['id'];
        $stmt = $pdo->prepare("UPDATE convidados SET nome = NULL, confirmado = 0, presente = 0, data_checkin = NULL WHERE id = :id");
        $stmt->execute([':id' => $id]);
        $msg_sucesso = "✅ Ingresso #" . formatarNumero($id) . " liberado!";
    }
    
    if (isset($_POST['acao']) && $_POST['acao'] === 'deletar') {
        $id = $_POST['id'];
        $stmt = $pdo->prepare("SELECT qrcode_image FROM convidados WHERE id = :id");
        $stmt->execute([':id' => $id]);
        $conv = $stmt->fetch();
        if ($conv && $conv['qrcode_image'] && file_exists('qrcodes/' . $conv['qrcode_image'])) {
            unlink('qrcodes/' . $conv['qrcode_image']);
        }
        $stmt = $pdo->prepare("DELETE FROM convidados WHERE id = :id");
        $stmt->execute([':id' => $id]);
        $msg_sucesso = "✅ Ingresso #" . formatarNumero($id) . " deletado!";
    }
}

$filtro_status = $_GET['status'] ?? 'todos';
$busca = $_GET['busca'] ?? '';
$ordem = $_GET['ordem'] ?? 'id';
$where = [];
$params = [];

if ($filtro_status === 'disponivel') $where[] = "nome IS NULL";
elseif ($filtro_status === 'atribuido') $where[] = "nome IS NOT NULL AND confirmado = 0";
elseif ($filtro_status === 'confirmado') $where[] = "confirmado = 1 AND presente = 0";
elseif ($filtro_status === 'presente') $where[] = "presente = 1";

if (!empty($busca)) {
    $where[] = "(id LIKE :busca OR nome LIKE :busca)";
    $params[':busca'] = '%' . $busca . '%';
}

$where_sql = !empty($where) ? 'WHERE ' . implode(' AND ', $where) : '';
$ordem_sql = match($ordem) {
    'nome' => 'nome ASC',
    'confirmado' => 'confirmado DESC, id ASC',
    'recente' => 'data_cadastro DESC',
    default => 'id ASC'
};

$stmt = $pdo->prepare("SELECT * FROM convidados $where_sql ORDER BY $ordem_sql");
$stmt->execute($params);
$convidados = $stmt->fetchAll();

$stats = $pdo->query("
    SELECT 
        COUNT(*) as total,
        SUM(CASE WHEN nome IS NULL THEN 1 ELSE 0 END) as disponiveis,
        SUM(CASE WHEN nome IS NOT NULL AND confirmado = 0 THEN 1 ELSE 0 END) as atribuidos,
        SUM(CASE WHEN confirmado = 1 THEN 1 ELSE 0 END) as confirmados,
        SUM(CASE WHEN presente = 1 THEN 1 ELSE 0 END) as presentes,
        SUM(CASE WHEN compor_mesa = 1 AND nome IS NOT NULL THEN 1 ELSE 0 END) as compor_mesa
    FROM convidados
")->fetch();
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Painel Administrativo</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary-color: #0a1465;
            --primary-dark: #081050;
            --primary-light: #1a2b9e;
            --text-light: #ffffff;
            --text-dark: #333333;
            --bg-light: #f8f9fa;
            --bg-white: #ffffff;
            --border-color: #dee2e6;
            --success-color: #198754;
            --danger-color: #dc3545;
            --warning-color: #ffc107;
            --info-color: #0dcaf0;
            --whatsapp-color: #25D366;
            --transition-speed: 0.3s;
            --border-radius: 8px;
            --shadow: 0 2px 4px rgba(0,0,0,0.1);
            --shadow-lg: 0 4px 8px rgba(0,0,0,0.15);
            --shadow-xl: 0 8px 16px rgba(0,0,0,0.2);
        }

        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; background: var(--bg-light); }
        
        .header { background: linear-gradient(135deg, var(--primary-color) 0%, var(--primary-dark) 100%); color: var(--text-light); padding: 20px; box-shadow: var(--shadow); }
        .header-content { max-width: 1400px; margin: 0 auto; display: flex; justify-content: space-between; align-items: center; flex-wrap: wrap; gap: 15px; }
        .header-left h1 { font-size: 24px; margin-bottom: 5px; }
        .header-left p { font-size: 12px; opacity: 0.9; }
        .header-actions { display: flex; gap: 10px; flex-wrap: wrap; }
        .btn-header { padding: 10px 20px; background: rgba(255,255,255,0.2); color: white; text-decoration: none; border-radius: 8px; font-size: 14px; transition: all 0.3s; border: none; cursor: pointer; display: flex; align-items: center; gap: 8px; }
        .btn-header:hover { background: rgba(255,255,255,0.3); }
        .btn-header:disabled { opacity: 0.5; cursor: not-allowed; }
        .btn-logout { background: rgba(220, 53, 69, 0.8) !important; }
        .btn-logout:hover { background: rgba(220, 53, 69, 1) !important; }
        
        .container { max-width: 1400px; margin: 0 auto; padding: 20px; }
        
        .stats { display: grid; grid-template-columns: repeat(auto-fit, minmax(160px, 1fr)); gap: 20px; margin-bottom: 30px; }
        .stat-card { background: white; padding: 20px; border-radius: 12px; box-shadow: 0 2px 8px rgba(0,0,0,0.1); }
        .stat-card h3 { color: #666; font-size: 14px; margin-bottom: 10px; }
        .stat-card .number { font-size: 32px; font-weight: bold; color: var(--primary-color); }
        .stat-card.disponivel .number { color: #28a745; }
        .stat-card.atribuido .number { color: #ffc107; }
        .stat-card.confirmado .number { color: #17a2b8; }
        .stat-card.presente .number { color: #667eea; }
        .stat-card.mesa .number { color: #e83e8c; }
        
        .filters { background: var(--bg-white); padding: 20px; border-radius: var(--border-radius); box-shadow: var(--shadow); margin-bottom: 20px; }
        .filters-grid { display: grid; grid-template-columns: 1fr 1fr 1fr 100px; gap: 15px; align-items: end; }
        .form-group { display: flex; flex-direction: column; }
        .form-group label { margin-bottom: 5px; color: #666; font-size: 14px; }
        .form-group select, .form-group input { padding: 10px; border: 1px solid #ddd; border-radius: 8px; font-size: 14px; }
        .btn { padding: 10px 20px; border: none; border-radius: 8px; cursor: pointer; font-size: 14px; transition: all 0.3s; }
        .btn-primary { background: var(--primary-color); color: var(--text-light); }
        .btn-primary:hover { background: var(--primary-light); }
        
        .table-container { background: var(--bg-white); border-radius: var(--border-radius); box-shadow: var(--shadow); overflow-x: auto; }
        table { width: 100%; border-collapse: collapse; }
        thead { background: var(--bg-light); }
        th { padding: 15px; text-align: left; font-weight: 600; color: var(--text-dark); border-bottom: 2px solid var(--border-color); }
        td { padding: 15px; border-bottom: 1px solid var(--border-color); }
        tbody tr:hover { background: var(--bg-light); }
        
        .numero-badge { display: inline-block; background: var(--primary-light); color: white; padding: 5px 12px; border-radius: 20px; font-weight: bold; font-size: 14px; }
        .status-badge { display: inline-block; padding: 5px 12px; border-radius: 20px; font-size: 12px; font-weight: 600; }
        .status-disponivel { background: #d4edda; color: var(--success-color); }
        .status-atribuido { background: #fff3cd; color: var(--warning-color); }
        .status-confirmado { background: #d1ecf1; color: var(--info-color); }
        .status-presente { background: #d6d8ff; color: #2e3192; }
        .mesa-badge { display: inline-block; padding: 3px 8px; border-radius: 12px; font-size: 11px; font-weight: 600; background: #ffe0f0; color: #d63384; }
        
        .action-buttons { display: flex; gap: 8px; flex-wrap: wrap; }
        .btn-icon { 
            width: 40px; 
            height: 40px; 
            border-radius: 50%; 
            border: none; 
            cursor: pointer; 
            display: inline-flex; 
            align-items: center; 
            justify-content: center;
            transition: all 0.3s;
            font-size: 16px;
            color: white;
        }
        .btn-icon:hover { 
            transform: scale(1.1); 
            box-shadow: 0 4px 8px rgba(0,0,0,0.2);
        }
        .btn-icon.edit,
        .btn-icon.link,
        .btn-icon.clean,
        .btn-icon.delete { 
            background: var(--primary-light);
        }
        .btn-icon.whatsapp { 
            background: var(--whatsapp-color);
        }
        
        .modal { display: none; position: fixed; z-index: 1000; left: 0; top: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); }
        .modal-content { background: var(--bg-white); margin: 50px auto; padding: 30px; border-radius: var(--border-radius); max-width: 600px; box-shadow: var(--shadow-lg); }
        .modal-header { margin-bottom: 20px; display: flex; justify-content: space-between; align-items: center; }
        .modal-header h2 { color: #333; }
        .modal-close { font-size: 28px; font-weight: bold; cursor: pointer; color: #999; }
        .modal-close:hover { color: #333; }
        .form-control { width: 100%; padding: 12px; margin-bottom: 15px; border: 1px solid var(--border-color); border-radius: var(--border-radius); font-size: 14px; }
        .form-label { display: block; margin-bottom: 5px; color: #666; font-weight: 500; }
        .checkbox-group { display: flex; align-items: center; margin-bottom: 20px; padding: 15px; background: var(--bg-light); border-radius: var(--border-radius); }
        .checkbox-group input[type="checkbox"] { width: 20px; height: 20px; margin-right: 10px; cursor: pointer; }
        .checkbox-group label { cursor: pointer; font-size: 14px; color: #333; }
        
        .alert { padding: 15px 20px; border-radius: var(--border-radius); margin-bottom: 20px; }
        .alert-success { background: #d4edda; color: #155724; border: 1px solid #c3e6cb; }
        .alert-error { background: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; }
        
        .nome-vazio { color: #999; font-style: italic; }
        .qrcode-preview { text-align: center; margin: 20px 0; }
        .qrcode-preview img { max-width: 300px; border: 2px solid #ddd; border-radius: 8px; padding: 10px; }
        .link-box { background: #f8f9fa; padding: 15px; border-radius: 8px; margin: 15px 0; word-break: break-all; }
        .btn-copy { background: var(--success-color); color: var(--text-light); width: 100%; margin-top: 10px; }
        .btn-copy:hover { background: #198754; }
        .select-all-bar { background: #e7f3ff; padding: 15px 20px; border-radius: 8px; margin-bottom: 15px; display: flex; justify-content: space-between; align-items: center; }
        .select-checkbox { width: 18px; height: 18px; cursor: pointer; }
        
        .whatsapp-preview { background: #f0f0f0; padding: 15px; border-radius: 8px; margin: 15px 0; font-family: 'Segoe UI', Arial, sans-serif; }
        .whatsapp-preview p { margin: 10px 0; line-height: 1.6; color: #333; white-space: pre-wrap; }
        .phone-input-group { position: relative; }
        .phone-input-group i { position: absolute; left: 12px; top: 50%; transform: translateY(-50%); color: #666; }
        .phone-input-group input { padding-left: 40px; }
        
        @media (max-width: 768px) {
            .header-content { flex-direction: column; }
            .filters-grid { grid-template-columns: 1fr; }
            table { font-size: 12px; }
            th, td { padding: 10px; }
            .btn-icon { width: 35px; height: 35px; font-size: 14px; }
        }
    </style>
</head>
<body>
    <div class="header">
        <div class="header-content">
            <div class="header-left">
                <h1>Painel Administrativo - QR Codes</h1>
                <p> <?= getNomeUsuario() ?> | Evento Posse 2026</p>
            </div>
            <div class="header-actions">
                <a href="gerar.php" class="btn-header">
                    <i class="fas fa-plus"></i> Gerar Mais
                </a>
                <button onclick="baixarSelecionados()" class="btn-header" style="background: #28a745;" id="btnBaixar" disabled>
                    <i class="fas fa-download"></i> Baixar (<span id="countBaixar">0</span>)
                </button>
                <button onclick="imprimirSelecionados()" class="btn-header" style="background: #17a2b8;" id="btnImprimir" disabled>
                    <i class="fas fa-print"></i> Imprimir (<span id="countImprimir">0</span>)
                </button>
                <button onclick="apagarSelecionados()" class="btn-header" style="background: #dc3545;" id="btnApagar" disabled>
                    <i class="fas fa-trash"></i> Apagar (<span id="countApagar">0</span>)
                </button>
                <a href="logout.php" class="btn-header btn-logout" onclick="return confirm('Deseja realmente sair do sistema?')">
                    <i class="fas fa-sign-out-alt"></i> Sair
                </a>
            </div>
        </div>
    </div>
    
    <div class="container">
        <?php if (isset($msg_sucesso)): ?>
            <div class="alert alert-success"><?= $msg_sucesso ?></div>
        <?php endif; ?>
        
        <?php if (isset($msg_erro)): ?>
            <div class="alert alert-error"><?= $msg_erro ?></div>
        <?php endif; ?>
        
        <div class="stats">
            <div class="stat-card">
                <h3>Total</h3>
                <div class="number"><?= $stats['total'] ?></div>
            </div>
            <div class="stat-card disponivel">
                <h3>Disponíveis</h3>
                <div class="number"><?= $stats['disponiveis'] ?></div>
            </div>
            <div class="stat-card atribuido">
                <h3>Atribuídos</h3>
                <div class="number"><?= $stats['atribuidos'] ?></div>
            </div>
            <div class="stat-card confirmado">
                <h3>Confirmados</h3>
                <div class="number"><?= $stats['confirmados'] ?></div>
            </div>
            <div class="stat-card presente">
                <h3>Presentes</h3>
                <div class="number"><?= $stats['presentes'] ?></div>
            </div>
            <div class="stat-card mesa">
                <h3>Compor Mesa</h3>
                <div class="number"><?= $stats['compor_mesa'] ?></div>
            </div>
        </div>
        
        <div class="filters">
            <form method="GET" class="filters-grid">
                <div class="form-group">
                    <label>Status</label>
                    <select name="status">
                        <option value="todos" <?= $filtro_status === 'todos' ? 'selected' : '' ?>>Todos</option>
                        <option value="disponivel" <?= $filtro_status === 'disponivel' ? 'selected' : '' ?>>Disponíveis</option>
                        <option value="atribuido" <?= $filtro_status === 'atribuido' ? 'selected' : '' ?>>Atribuídos</option>
                        <option value="confirmado" <?= $filtro_status === 'confirmado' ? 'selected' : '' ?>>Confirmados</option>
                        <option value="presente" <?= $filtro_status === 'presente' ? 'selected' : '' ?>>Presentes</option>
                    </select>
                </div>
                <div class="form-group">
                    <label>Buscar</label>
                    <input type="text" name="busca" placeholder="Número ou nome..." value="<?= htmlspecialchars($busca) ?>">
                </div>
                <div class="form-group">
                    <label>Ordenar</label>
                    <select name="ordem">
                        <option value="id" <?= $ordem === 'id' ? 'selected' : '' ?>>Número</option>
                        <option value="nome" <?= $ordem === 'nome' ? 'selected' : '' ?>>Nome</option>
                        <option value="confirmado" <?= $ordem === 'confirmado' ? 'selected' : '' ?>>Confirmados</option>
                        <option value="recente" <?= $ordem === 'recente' ? 'selected' : '' ?>>Mais recente</option>
                    </select>
                </div>
                <button type="submit" class="btn btn-primary">Filtrar</button>
            </form>
        </div>
        
        <div class="select-all-bar">
            <div>
                <input type="checkbox" id="selectAll" class="select-checkbox" onchange="selecionarTodos(this)">
                <label for="selectAll" style="cursor: pointer; margin-left: 10px; font-weight: 600;">Selecionar Todos</label>
            </div>
            <span style="color: #666; font-size: 14px;">Baixar, imprimir ou apagar os selecionados</span>
        </div>
        
        <div class="table-container">
            <table>
                <thead>
                    <tr>
                        <th width="50"><input type="checkbox" class="select-checkbox" onchange="selecionarTodos(this)"></th>
                        <th>Número</th>
                        <th>Nome</th>
                        <th>Status</th>
                        <th>Mesa</th>
                        <th>Ações</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($convidados as $conv): ?>
                    <tr>
                        <td>
                            <input type="checkbox" class="qr-checkbox" value="<?= $conv['id'] ?>" onchange="atualizarContadores()">
                        </td>
                        <td><span class="numero-badge">#<?= formatarNumero($conv['id']) ?></span></td>
                        <td>
                            <?php if (empty($conv['nome'])): ?>
                                <span class="nome-vazio">Disponível</span>
                            <?php else: ?>
                                <?= htmlspecialchars($conv['nome']) ?>
                            <?php endif; ?>
                        </td>
                        <td>
                            <?php
                            if ($conv['presente']) echo '<span class="status-badge status-presente">✓ Presente</span>';
                            elseif ($conv['confirmado']) echo '<span class="status-badge status-confirmado">✓ Confirmado</span>';
                            elseif (!empty($conv['nome'])) echo '<span class="status-badge status-atribuido">⏳ Atribuído</span>';
                            else echo '<span class="status-badge status-disponivel">⚪ Disponível</span>';
                            ?>
                        </td>
                        <td>
                            <?php if (!empty($conv['nome']) && $conv['compor_mesa']): ?>
                                <span class="mesa-badge">👥 Mesa</span>
                            <?php endif; ?>
                        </td>
                        <td>
                            <div class="action-buttons">
                                <button onclick="editarIngresso(<?= htmlspecialchars(json_encode($conv)) ?>)" class="btn-icon edit" title="Editar">
                                    <i class="fas fa-pencil-alt"></i>
                                </button>
                                <button onclick="mostrarQRCode(<?= htmlspecialchars(json_encode($conv)) ?>)" class="btn-icon link" title="Abrir Link">
                                    <i class="fas fa-link"></i>
                                </button>
                                <button onclick="enviarWhatsApp(<?= htmlspecialchars(json_encode($conv)) ?>)" class="btn-icon whatsapp" title="Enviar no WhatsApp">
                                    <i class="fab fa-whatsapp"></i>
                                </button>
                                <?php if (!empty($conv['nome'])): ?>
                                <button onclick="limparIngresso(<?= $conv['id'] ?>)" class="btn-icon clean" title="Limpar">
                                    <i class="fas fa-broom"></i>
                                </button>
                                <?php endif; ?>
                                <button onclick="deletarIngresso(<?= $conv['id'] ?>)" class="btn-icon delete" title="Deletar">
                                    <i class="fas fa-trash-alt"></i>
                                </button>
                            </div>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
    
    <!-- Modal Editar -->
    <div id="modalEditar" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h2 id="modalTitulo">Editar Ingresso</h2>
                <span class="modal-close" onclick="fecharModal('modalEditar')">&times;</span>
            </div>
            <form method="POST">
                <input type="hidden" name="acao" value="atribuir">
                <input type="hidden" name="id" id="editId">
                <label class="form-label">Nome do Convidado:</label>
                <input type="text" name="nome" id="editNome" class="form-control" placeholder="Digite o nome completo" required>
                <div class="checkbox-group">
                    <input type="checkbox" name="compor_mesa" id="editMesa" value="1">
                    <label for="editMesa">👥 <strong>Compor a Mesa Diretora</strong> do evento</label>
                </div>
                <button type="submit" class="btn btn-primary" style="width: 100%;">
                    <i class="fas fa-save"></i> Salvar
                </button>
            </form>
        </div>
    </div>
    
    <!-- Modal QR Code -->
    <div id="modalQR" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h2 id="qrTitulo">QR Code</h2>
                <span class="modal-close" onclick="fecharModal('modalQR')">&times;</span>
            </div>
            <div class="qrcode-preview">
                <img id="qrImagem" src="" alt="QR Code">
            </div>
            <div class="link-box">
                <strong>Link:</strong><br>
                <span id="qrLink"></span>
            </div>
            <button onclick="copiarLink()" class="btn btn-copy">
                <i class="fas fa-copy"></i> Copiar Link
            </button>
        </div>
    </div>
    
    <!-- Modal WhatsApp -->
    <div id="modalWhatsApp" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h2 id="whatsappTitulo">
                    <i class="fab fa-whatsapp" style="color: #25D366;"></i> Enviar Convite via WhatsApp
                </h2>
                <span class="modal-close" onclick="fecharModal('modalWhatsApp')">&times;</span>
            </div>
            <div style="margin-bottom: 20px;">
                <label class="form-label">
                    <i class="fas fa-phone"></i> Número do WhatsApp (com DDD):
                </label>
                <div class="phone-input-group">
                    <i class="fas fa-mobile-alt"></i>
                    <input type="tel" id="whatsappNumero" class="form-control" placeholder="Ex: 62999999999" maxlength="15">
                </div>
                <small style="color: #666; display: block; margin-top: 5px;">
                    Digite apenas números (DDD + número). Ex: 62999999999
                </small>
            </div>
            <div class="whatsapp-preview">
                <strong style="display: block; margin-bottom: 10px;">
                    <i class="fas fa-eye"></i> Prévia da mensagem:
                </strong>
                <p id="whatsappMensagemPreview"></p>
            </div>
            <button onclick="confirmarEnvioWhatsApp()" class="btn" style="width: 100%; background: #25D366; color: white; font-size: 16px; padding: 15px;">
                <i class="fab fa-whatsapp"></i> Enviar via WhatsApp
            </button>
        </div>
    </div>
    
    <script>
        function editarIngresso(conv) {
            document.getElementById('modalTitulo').textContent = 'Editar Ingresso #' + conv.id.toString().padStart(4, '0');
            document.getElementById('editId').value = conv.id;
            document.getElementById('editNome').value = conv.nome || '';
            document.getElementById('editMesa').checked = conv.compor_mesa == 1;
            document.getElementById('modalEditar').style.display = 'block';
        }
        
        function mostrarQRCode(conv) {
            const url = '<?= URL_BASE ?>/confirmar.php?id=' + conv.id;
            document.getElementById('qrTitulo').textContent = 'QR Code #' + conv.id.toString().padStart(4, '0');
            document.getElementById('qrImagem').src = 'qrcodes/' + conv.qrcode_image;
            document.getElementById('qrLink').textContent = url;
            window.currentLink = url;
            document.getElementById('modalQR').style.display = 'block';
        }
        
        function enviarWhatsApp(conv) {
            const url = '<?= URL_BASE ?>/confirmar.php?id=' + conv.id;
            const numero = conv.id.toString().padStart(4, '0');
            const nomeConvidado = conv.nome || 'Convidado';
            const comporMesa = conv.compor_mesa == 1;
            
            window.currentWhatsAppLink = url;
            window.currentWhatsAppNome = nomeConvidado;
            window.currentWhatsAppComporMesa = comporMesa;
            
            document.getElementById('whatsappTitulo').innerHTML = '<i class="fab fa-whatsapp" style="color: #25D366;"></i> Enviar Convite #' + numero + ' via WhatsApp';
            document.getElementById('whatsappNumero').value = '';
            
            // Mensagem diferente para quem vai compor mesa
            let mensagem;
            if (comporMesa) {
                mensagem = `CONVITE – SOLENIDADE DE POSSE ASSEGO 2026/2029

O Subtenente Sérgio, Presidente reeleito da Associação dos Subtenentes e Sargentos do Estado de Goiás (ASSEGO), tem a honra de convidar  ${nomeConvidado} para compor a Mesa Diretora da Solenidade de Posse da Diretoria Executiva e do Conselho Fiscal – Gestão 2026/2029.

Para confirmar sua presença, acesse o link abaixo:
${url}`;
            } else {
                mensagem = `CONVITE – SOLENIDADE DE POSSE ASSEGO 2026/2029

O Subtenente Sérgio, Presidente reeleito da Associação dos Subtenentes e Sargentos do Estado de Goiás (ASSEGO), tem a honra de convidar ${nomeConvidado} para a Solenidade de Posse da Diretoria Executiva e do Conselho Fiscal – Gestão 2026/2029.

Para confirmar sua presença, acesse o link abaixo:
${url}`;
            }
            
            document.getElementById('whatsappMensagemPreview').textContent = mensagem;
            document.getElementById('modalWhatsApp').style.display = 'block';
        }
        
        function confirmarEnvioWhatsApp() {
            const numero = document.getElementById('whatsappNumero').value.replace(/\D/g, '');
            
            if (!numero) {
                alert('❌ Por favor, digite o número do WhatsApp!');
                document.getElementById('whatsappNumero').focus();
                return;
            }
            
            if (numero.length < 10) {
                alert('❌ Número inválido! Digite o DDD + número completo.');
                document.getElementById('whatsappNumero').focus();
                return;
            }
            
            const nomeConvidado = window.currentWhatsAppNome;
            const link = window.currentWhatsAppLink;
            const comporMesa = window.currentWhatsAppComporMesa;
            
            // Mensagem diferente para quem vai compor mesa
            let mensagem;
            if (comporMesa) {
                mensagem = `CONVITE – SOLENIDADE DE POSSE ASSEGO 2026/2029

O Subtenente Sérgio, Presidente reeleito da Associação dos Subtenentes e Sargentos do Estado de Goiás (ASSEGO), tem a honra de convidar o Excelentíssimo Senhor ${nomeConvidado} para compor a Mesa Diretora da Solenidade de Posse da Diretoria Executiva e do Conselho Fiscal – Gestão 2026/2029.

Para confirmar sua presença, acesse o link abaixo:
${link}`;
            } else {
                mensagem = `CONVITE – SOLENIDADE DE POSSE ASSEGO 2026/2029

O Subtenente Sérgio, Presidente reeleito da Associação dos Subtenentes e Sargentos do Estado de Goiás (ASSEGO), tem a honra de convidar ${nomeConvidado} para a Solenidade de Posse da Diretoria Executiva e do Conselho Fiscal – Gestão 2026/2029.

Para confirmar sua presença, acesse o link abaixo:
${link}`;
            }
            
            const mensagemEncoded = encodeURIComponent(mensagem);
            const numeroFormatado = '55' + numero;
            const whatsappUrl = `https://api.whatsapp.com/send?phone=${numeroFormatado}&text=${mensagemEncoded}`;
            
            window.open(whatsappUrl, '_blank');
            fecharModal('modalWhatsApp');
            
            setTimeout(() => {
                alert('✅ WhatsApp aberto! Confira a mensagem e clique em enviar.');
            }, 500);
        }
        
        function fecharModal(id) { 
            document.getElementById(id).style.display = 'none'; 
        }
        
        function limparIngresso(id) {
            if (confirm('Limpar este ingresso? O nome e confirmação serão removidos.')) {
                const form = document.createElement('form');
                form.method = 'POST';
                form.innerHTML = '<input type="hidden" name="acao" value="limpar"><input type="hidden" name="id" value="' + id + '">';
                document.body.appendChild(form);
                form.submit();
            }
        }
        
        function deletarIngresso(id) {
            if (confirm('⚠️ DELETAR PERMANENTEMENTE ESTE QR CODE?\n\nEsta ação não pode ser desfeita!')) {
                const form = document.createElement('form');
                form.method = 'POST';
                form.innerHTML = '<input type="hidden" name="acao" value="deletar"><input type="hidden" name="id" value="' + id + '">';
                document.body.appendChild(form);
                form.submit();
            }
        }
        
        function copiarLink() {
            const link = window.currentLink;
            if (navigator.clipboard && navigator.clipboard.writeText) {
                navigator.clipboard.writeText(link).then(() => alert('✓ Link copiado!')).catch(() => copiarFallback(link));
            } else {
                copiarFallback(link);
            }
        }
        
        function copiarFallback(text) {
            const textarea = document.createElement('textarea');
            textarea.value = text;
            textarea.style.position = 'fixed';
            textarea.style.opacity = '0';
            document.body.appendChild(textarea);
            textarea.select();
            try {
                document.execCommand('copy');
                alert('✓ Link copiado!');
            } catch (err) {
                alert('❌ Erro ao copiar');
            }
            document.body.removeChild(textarea);
        }
        
        function selecionarTodos(checkbox) {
            const checkboxes = document.querySelectorAll('.qr-checkbox');
            checkboxes.forEach(cb => cb.checked = checkbox.checked);
            atualizarContadores();
        }
        
        function atualizarContadores() {
            const checked = document.querySelectorAll('.qr-checkbox:checked').length;
            document.getElementById('countImprimir').textContent = checked;
            document.getElementById('countBaixar').textContent = checked;
            document.getElementById('countApagar').textContent = checked;
            document.getElementById('btnImprimir').disabled = checked === 0;
            document.getElementById('btnBaixar').disabled = checked === 0;
            document.getElementById('btnApagar').disabled = checked === 0;
        }
        
        function imprimirSelecionados() {
            const checkboxes = document.querySelectorAll('.qr-checkbox:checked');
            if (checkboxes.length === 0) {
                alert('❌ Selecione pelo menos 1 QR code!');
                return;
            }
            const ids = Array.from(checkboxes).map(cb => cb.value);
            window.open('imprimir_massa.php?ids=' + ids.join(','), '_blank');
        }
        
        function baixarSelecionados() {
            const checkboxes = document.querySelectorAll('.qr-checkbox:checked');
            if (checkboxes.length === 0) {
                alert('❌ Selecione pelo menos 1 QR code!');
                return;
            }
            const ids = Array.from(checkboxes).map(cb => cb.value);
            if (!confirm('Deseja baixar ' + ids.length + ' QR codes em um arquivo ZIP?')) return;
            const form = document.createElement('form');
            form.method = 'POST';
            form.innerHTML = '<input type="hidden" name="acao" value="download_massa"><input type="hidden" name="ids" value=\'' + JSON.stringify(ids) + '\'>';
            document.body.appendChild(form);
            form.submit();
            document.body.removeChild(form);
        }
        
        function apagarSelecionados() {
            const checkboxes = document.querySelectorAll('.qr-checkbox:checked');
            if (checkboxes.length === 0) {
                alert('❌ Selecione pelo menos 1 QR code!');
                return;
            }
            const ids = Array.from(checkboxes).map(cb => cb.value);
            if (!confirm('⚠️ ATENÇÃO!\n\nVocê está prestes a DELETAR PERMANENTEMENTE ' + ids.length + ' QR code(s).\n\nEsta ação NÃO pode ser desfeita!\n\nDeseja continuar?')) return;
            const form = document.createElement('form');
            form.method = 'POST';
            form.innerHTML = '<input type="hidden" name="acao" value="apagar_massa"><input type="hidden" name="ids" value=\'' + JSON.stringify(ids) + '\'>';
            document.body.appendChild(form);
            form.submit();
        }
        
        window.onclick = function(event) {
            if (event.target.className === 'modal') event.target.style.display = 'none';
        }
        
        document.addEventListener('DOMContentLoaded', function() {
            atualizarContadores();
        });
    </script>
</body>
</html>